/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2002 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * dxcc.c - dxcc lookups and creation of the hashtable
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dxcc.h"
#include "gc.h"
#include "support.h"
#include "types.h"

extern GtkWidget *mainwindow;
extern preferencestype preferences;
extern statetype state;
gchar **dxcc;
GHashTable *prefixes = NULL;

/* if there is an underscore in the prefix this means we	have to substitute 0-9 */
static gboolean find_underscore(gchar *string, gchar *primaryprefix, guint i) {
	gchar **underscore = NULL, *end, *j;
	guint k;

	underscore = g_strsplit(string, "_", 0);
	if (underscore[1] != NULL) {
		end = string + strlen(string);
		for (j = string; j < end; ++j) {
			switch (*j) {
				case '_':
					for (k = 0; k <= 9; k++) {
						*j = 48 + k;
						g_hash_table_insert(prefixes, g_strdup(string), GINT_TO_POINTER(i + 1));
						state.prefixes++;
					}
				break;
			}
		}
		g_strfreev(underscore);
		return(TRUE);
	}
	else {
		g_strfreev(underscore);
		return(FALSE);
	}
}

/* see if we can find a string like 4J-4K and substitute the prefixes in between */
static gboolean find_dash(gchar *string, gchar *primaryprefix, guint i) {
	gchar **dash = NULL, *begin, *end, *common, *newprefix;
	gchar first, j;

	dash = g_strsplit(string, "-", 0);
	if (dash[1] != NULL) {						/*4J-4K*/
		begin = dash[0] + 1;					/* J */
		first = *begin;
		end = dash[1] + 1;						/* K */
		common = g_strndup(dash[0], 1);	/* 4 */
		for (j = *begin; j <= *end; j++) {
			newprefix = g_strdup_printf("%s%c", common, j);
			if (g_strcasecmp(newprefix, primaryprefix) != 0) { /* only add new prefixes */
				g_hash_table_insert(prefixes, g_strdup(newprefix), GINT_TO_POINTER(i + 1));
				state.prefixes++;
			}
		}
		if (j - first == 26) /* we have A-Z, so first character also needs to be added */
			if (g_strcasecmp(common, primaryprefix) != 0) {
				g_hash_table_insert(prefixes, g_strdup(common), GINT_TO_POINTER(i + 1));
				state.prefixes++;
			}
		g_free(common);
		g_strfreev(dash);
		return(TRUE);
	}
	else {
		g_strfreev(dash);
		return(FALSE);
	}
}

static void convert_to_hashtable(void) {
	guint i, ITUindex = 0, otherindex = 0;
	gchar **dxccsplit = NULL, *otherprefix = NULL, *reverseprimaryprefix = NULL,
		**ITUsplit = NULL, **othersplit = NULL;

	/* we need fields 0, 8 and 9 which are primary, ITU and other prefixes */
	for (i = 0; i <= state.countries; i++)
	{
		dxccsplit = g_strsplit(dxcc[i], ":", 0);

		/* copy first field and reverse it for looking up deleted countries (marked by '*') */
		reverseprimaryprefix = g_strdup(dxccsplit[0]);
		g_strreverse(reverseprimaryprefix);

		/* skip deleted countries */
		if (g_strncasecmp(reverseprimaryprefix, "*", 1) != 0)
		{
			/* only add first field if not empty */
			if (g_strcasecmp(dxccsplit[0], "") != 0)
			{
				/* check for underscore in primary prefix */
				if (!find_underscore(dxccsplit[0], dxccsplit[0], i))
				{
					g_hash_table_insert(prefixes, g_strdup(dxccsplit[0]), GINT_TO_POINTER(i + 1));
					state.prefixes++;
				}
			}
			/* check out ITU prefix allocations */
			if (g_strcasecmp(dxccsplit[8], "") != 0)
			{
				/* see of we can find a ',' */
				ITUsplit = g_strsplit(dxccsplit[8], ",", 0);
				for (;;)
				{
					if (ITUsplit[ITUindex] == NULL) break;
					/* see of we can find a '-' */
					if (!find_dash(ITUsplit[ITUindex], dxccsplit[0], i))
						/* no dashes, just add the prefix */
						if (g_strcasecmp(ITUsplit[ITUindex], dxccsplit[0]) != 0)
						{
							g_hash_table_insert(prefixes, g_strdup(ITUsplit[ITUindex]),
								GINT_TO_POINTER(i + 1));
							state.prefixes++;
						}
					ITUindex++;
				}
				ITUindex = 0;
			}
			/* check out other prefixes */
			/* remove linefeed in last field (always contains a linefeed) */
			otherprefix = g_strndup(dxccsplit[9], strlen(dxccsplit[9])-1);
			if (g_strcasecmp(otherprefix, "") != 0)
			{
				othersplit = g_strsplit(otherprefix, ",", 0);
				for (;;)
				{
					if (othersplit[otherindex] == NULL) break;
					if (!find_dash(othersplit[otherindex], dxccsplit[0], i))
						if (!find_underscore(othersplit[otherindex], dxccsplit[0], i))
							if (g_strcasecmp(othersplit[otherindex], dxccsplit[0]) != 0)
							{
								g_hash_table_insert(prefixes, g_strdup(othersplit[otherindex]), 
									GINT_TO_POINTER(i + 1));
								state.prefixes++;
							}
					otherindex++;
				}
				otherindex = 0;
			}
		}
	}
	g_free(reverseprimaryprefix);
	g_free(otherprefix);
	g_strfreev(othersplit);
	g_strfreev(ITUsplit);
	g_strfreev(dxccsplit);
}

void loaddxcc(void) {
	gchar *dxccfile, line[200];
	FILE *fp;
	gint i;

	dxcc = g_new0(gchar *, 400);
	for (i = 0; i < 400; i++)
		dxcc[i] = g_new0(gchar, 200);

	dxccfile = g_strconcat(PACKAGE_DATA_DIR, G_DIR_SEPARATOR_S, "dxcc",
		G_DIR_SEPARATOR_S, "dxcck2di.txt", NULL);
	fp = fopen(dxccfile, "r");
	if (fp != NULL)
	{

		/*read the file until the list starts*/
		for(;;) {
			fgets(line, 200, fp);
			if (g_strncasecmp(line, "---", 3) == 0) break;
		}
		fgets(line, 200, fp); /*one extra line with dashes*/

		/* read all the lines and fill the array */
		for(;;) {
			fgets(line, 200, fp);
			if (g_strncasecmp(line, "---", 3) == 0 || state.countries == 400) { 
				state.countries--; 
				break; 
			}
			dxcc[state.countries++] = g_strdup(line);
		}
		fclose(fp);
		prefixes = g_hash_table_new(g_str_hash, g_str_equal);
		convert_to_hashtable();
	}
	g_free(dxccfile);
}

static gint lookup_dxcc(gchar *callsign) {
	gint result, country;
	gchar *prefix;

	prefix = g_strndup(callsign, 4);
	country = GPOINTER_TO_INT(g_hash_table_lookup(prefixes, prefix));
	g_free(prefix);

	if (country == 0) {
		prefix = g_strndup(callsign, 3);
		country = GPOINTER_TO_INT(g_hash_table_lookup(prefixes, prefix));
		g_free(prefix);
	}

	if (country == 0) {
		prefix = g_strndup(callsign, 2);
		country = GPOINTER_TO_INT(g_hash_table_lookup(prefixes, prefix));
		g_free(prefix);
	}

	if (country == 0) {
		prefix = g_strndup(callsign, 1);
		country = GPOINTER_TO_INT(g_hash_table_lookup(prefixes, prefix));
		g_free(prefix);
	}

	/* country starts at 1 */
	if (country == 0) result = -1; else result = country - 1;
	return(result);
}

void updatedxccframe(gchar *callsign) {
	GtkWidget *dxcclabel1, *dxcclabel2, *dxcclabel3, *dxcclabel4, 
		*dxcclabel5, *dxccframe;
	gchar **dxccsplit = NULL;
	gchar *labeltext1, *labeltext2, *labeltext3, *labeltext4;
	gint country = 0;
	gchar *gcresult;

	dxcclabel1 = lookup_widget(mainwindow, "dxcclabel1");
	dxcclabel2 = lookup_widget(mainwindow, "dxcclabel2");
	dxcclabel3 = lookup_widget(mainwindow, "dxcclabel3");
	dxcclabel4 = lookup_widget(mainwindow, "dxcclabel4");
	dxcclabel5 = lookup_widget(mainwindow, "dxcclabel5");
	dxccframe = lookup_widget(mainwindow, "dxccframe");
	labeltext1 = g_strdup("");
	labeltext2 = g_strdup("");
	labeltext3 = g_strdup("");
	labeltext4 = g_strdup("");
	gcresult = g_strdup("");
	gtk_frame_set_label(GTK_FRAME(dxccframe), "DXCC");

	if (g_strcasecmp(callsign, "") != 0) {
		country = lookup_dxcc(callsign);
		gtk_frame_set_label(GTK_FRAME(dxccframe), callsign);
	}

	if (country > 0) {
		dxccsplit = g_strsplit(dxcc[country], ":", 0);
		labeltext1 = g_strdup(dxccsplit[1]);
		labeltext2 = g_strdup_printf("%s - ITU %s - CQ %s", dxccsplit[2], dxccsplit[3], dxccsplit[4]);
		labeltext3 = g_strdup_printf(_("Timezone: %s"), dxccsplit[5]);
		labeltext4 = g_strdup_printf(_("Location: %s, %s"), dxccsplit[6], dxccsplit[7]);
		gcresult = g_strdup(gc(preferences.units, state.mylocation, g_strconcat(dxccsplit[6], dxccsplit[7], NULL)));
	}
	else if (country == -1) {
		labeltext1 = g_strdup(_("Unknown Country"));
	}

	gtk_label_set_text(GTK_LABEL(dxcclabel1), labeltext1);
	gtk_label_set_text(GTK_LABEL(dxcclabel2), labeltext2);
	gtk_label_set_text(GTK_LABEL(dxcclabel3), labeltext3);
	gtk_label_set_text(GTK_LABEL(dxcclabel4), labeltext4);
	gtk_label_set_text(GTK_LABEL(dxcclabel5), gcresult);
	g_free(labeltext1);
	g_free(labeltext2);
	g_free(labeltext3);
	g_free(labeltext4);
	g_free(gcresult);
	g_strfreev(dxccsplit);
}
